#region Includes
using GroundhogLearning.GameKit.Utils;
using UnityEngine;
using UnityEngine.Events;
#endregion

namespace GroundhogLearning.GameKit
{
    [AddComponentMenu(EditorMenuConfig.CATEGORY_TRANSFORMS + "Destroyable")]
    public class Destroyable : MonoBehaviour
    {
        #region Variables

        [Header("Configuration")]
        [SerializeField, Tooltip("If true, this object will destroy itself when the scene starts.")]
        private bool _destroyOnStart = false;

        [SerializeField, Tooltip("If true, this object destroys itself when it collides or triggers another object.")]
        private bool _destroyOnContact = true;

        [SerializeField, Tooltip("Delay (in seconds) before destroying the object.")]
        private float _destroyDelay = 0f;

        [Header("Events")]
        [SerializeField, Tooltip("Event raised when the object is destroyed")]
        private UnityEvent<GameObject> OnDestroyed;

        #endregion

        private void Start()
        {
            if (_destroyOnStart)
            {
                DestroyObject();
            }
        }

        private void OnCollisionEnter(Collision collision)
        {
            Debug.Log($"Destroyable | {gameObject.name} detected collision with {collision.collider.name}.");

            if (!_destroyOnContact) return;
            DestroyObject();
        }

        private void OnTriggerEnter(Collider other)
        {
            Debug.Log($"Destroyable | {gameObject.name} detected collision with {other.name}.");

            if (!_destroyOnContact) return;
            DestroyObject();
        }

        private void DestroyObject()
        {
            Debug.Log($"Destroyable | {gameObject.name} will be destroyed.");

            OnDestroyed?.Invoke(gameObject);
            Destroy(gameObject, _destroyDelay);
        }
    }
}
